@extends('layouts.app')

@php
    use Illuminate\Support\Facades\Storage;
    use Illuminate\Support\Str;
@endphp

@section('title', 'Manage Stores')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="mb-8 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Store Management</h1>
            <p class="text-gray-600 mt-2">Manage store registrations and approvals</p>
        </div>
        <a href="{{ route('admin.dashboard') }}" class="text-primary hover:text-primary-dark text-sm font-medium">
            ← Back to Dashboard
        </a>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6 rounded" role="alert">
            <p>{{ session('success') }}</p>
        </div>
    @endif

    @if(session('error'))
        <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6 rounded" role="alert">
            <p>{{ session('error') }}</p>
        </div>
    @endif

    <!-- Filters -->
    <div class="bg-white shadow rounded-lg p-6 mb-6">
        <form method="GET" action="{{ route('admin.stores.index') }}" class="grid grid-cols-1 md:grid-cols-4 gap-4">
            <!-- Status Filter -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                <select name="status" class="w-full rounded-md border-gray-300 shadow-sm focus:border-primary focus:ring-primary">
                    <option value="">All Stores ({{ $totalCount }})</option>
                    <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending ({{ $pendingCount }})</option>
                    <option value="approved" {{ request('status') == 'approved' ? 'selected' : '' }}>Approved ({{ $approvedCount }})</option>
                    <option value="suspended" {{ request('status') == 'suspended' ? 'selected' : '' }}>Suspended ({{ $suspendedCount }})</option>
                    <option value="rejected" {{ request('status') == 'rejected' ? 'selected' : '' }}>Rejected ({{ $rejectedCount }})</option>
                </select>
            </div>

            <!-- Search -->
            <div class="md:col-span-2">
                <label class="block text-sm font-medium text-gray-700 mb-2">Search</label>
                <input type="text" name="search" value="{{ request('search') }}" placeholder="Search by store name, owner name, or email..." 
                       class="w-full rounded-md border-gray-300 shadow-sm focus:border-primary focus:ring-primary">
            </div>

            <!-- Buttons -->
            <div class="flex items-end gap-2">
                <button type="submit" class="flex-1 bg-primary text-white px-4 py-2 rounded-md hover:bg-primary-dark transition font-medium">
                    Filter
                </button>
                <a href="{{ route('admin.stores.index') }}" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 transition font-medium">
                    Clear
                </a>
            </div>
        </form>
    </div>

    <!-- Stores List -->
    <div class="bg-white shadow overflow-hidden sm:rounded-md">
        @if($stores->count() > 0)
            <ul class="divide-y divide-gray-200">
                @foreach($stores as $store)
                    <li>
                        <div class="px-4 py-4 sm:px-6">
                            <div class="flex items-start justify-between">
                                <div class="flex-1 min-w-0">
                                    <div class="flex items-center gap-3 mb-2">
                                        @if($store->logo)
                                            <img src="{{ Storage::url($store->logo) }}" alt="{{ $store->business_name }}" 
                                                 class="h-12 w-12 rounded-lg object-cover">
                                        @else
                                            <div class="h-12 w-12 rounded-lg bg-gray-200 flex items-center justify-center">
                                                <svg class="w-6 h-6 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/>
                                                </svg>
                                            </div>
                                        @endif
                                        <div class="flex-1">
                                            <div class="flex items-center gap-2">
                                                <h3 class="text-lg font-semibold text-gray-900">{{ $store->business_name }}</h3>
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                                    {{ $store->status === 'approved' ? 'bg-green-100 text-green-800' : 
                                                       ($store->status === 'pending' ? 'bg-yellow-100 text-yellow-800' : 
                                                       ($store->status === 'suspended' ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-gray-800')) }}">
                                                    {{ ucfirst($store->status) }}
                                                </span>
                                                @if($store->is_featured)
                                                    <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium bg-purple-100 text-purple-800">
                                                        Featured
                                                    </span>
                                                @endif
                                            </div>
                                            @if($store->user)
                                                <p class="text-sm text-gray-600 mt-1">
                                                    Owner: <span class="font-medium">{{ $store->user->name }}</span> • {{ $store->user->email }}
                                                </p>
                                            @else
                                                <p class="text-sm text-gray-500 mt-1">Owner not found</p>
                                            @endif
                                        </div>
                                    </div>

                                    <!-- Store Details -->
                                    <div class="mt-3 grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-gray-600">
                                        @if($store->description)
                                            <div>
                                                <span class="font-medium">Description:</span> {{ Str::limit($store->description, 100) }}
                                            </div>
                                        @endif
                                        @if($store->phone)
                                            <div>
                                                <span class="font-medium">Phone:</span> {{ $store->phone }}
                                            </div>
                                        @endif
                                        @if($store->city || $store->state || $store->country)
                                            <div>
                                                <span class="font-medium">Location:</span> 
                                                {{ implode(', ', array_filter([$store->city, $store->state, $store->country])) }}
                                            </div>
                                        @endif
                                        <div>
                                            <span class="font-medium">Products:</span> {{ $store->products()->count() }}
                                        </div>
                                        <div>
                                            <span class="font-medium">Total Sales:</span> {{ currency($store->total_sales ?? 0) }}
                                        </div>
                                        <div>
                                            <span class="font-medium">Earnings:</span> {{ currency($store->total_earnings ?? 0) }}
                                        </div>
                                        @if($store->approved_at)
                                            <div>
                                                <span class="font-medium">Approved:</span> {{ $store->approved_at->format('M d, Y') }}
                                            </div>
                                        @endif
                                        <div>
                                            <span class="font-medium">Registered:</span> {{ $store->created_at->format('M d, Y') }}
                                        </div>
                                    </div>

                                    @if($store->rejection_reason)
                                        <div class="mt-2 p-2 bg-red-50 border border-red-200 rounded text-sm text-red-700">
                                            <span class="font-medium">Rejection/Suspension Reason:</span> {{ $store->rejection_reason }}
                                        </div>
                                    @endif

                                    <!-- Store Links -->
                                    <div class="mt-3 flex flex-wrap gap-2">
                                        @if($store->website)
                                            <a href="{{ $store->website }}" target="_blank" rel="noopener" 
                                               class="text-sm text-primary hover:text-primary-dark">
                                                Website →
                                            </a>
                                        @endif
                                        @if($store->status === 'approved')
                                            <a href="{{ route('store.show', $store->slug) }}" target="_blank" 
                                               class="text-sm text-primary hover:text-primary-dark">
                                                View Store →
                                            </a>
                                        @endif
                                    </div>
                                </div>

                                <!-- Actions -->
                                <div class="ml-4 flex flex-col gap-2">
                                    @if($store->status === 'pending')
                                        <form action="{{ route('admin.manual-approval.approve-store') }}" method="POST" class="inline">
                                            @csrf
                                            <input type="hidden" name="store_id" value="{{ $store->id }}">
                                            <button type="submit" 
                                                    onclick="return confirm('Approve this store?')"
                                                    class="w-full bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition text-sm font-medium">
                                                Approve
                                            </button>
                                        </form>

                                        <button type="button" 
                                                onclick="showRejectModal({{ $store->id }}, '{{ addslashes($store->business_name) }}')"
                                                class="w-full bg-red-600 text-white px-4 py-2 rounded-md hover:bg-red-700 transition text-sm font-medium">
                                            Reject
                                        </button>
                                    @endif

                                    @if($store->status === 'approved')
                                        <button type="button" 
                                                onclick="showSuspendModal({{ $store->id }}, '{{ addslashes($store->business_name) }}')"
                                                class="w-full bg-yellow-600 text-white px-4 py-2 rounded-md hover:bg-yellow-700 transition text-sm font-medium">
                                            Suspend
                                        </button>
                                    @endif

                                    @if($store->status === 'suspended')
                                        <form action="{{ route('admin.manual-approval.unsuspend-store') }}" method="POST" class="inline">
                                            @csrf
                                            <input type="hidden" name="store_id" value="{{ $store->id }}">
                                            <button type="submit" 
                                                    onclick="return confirm('Unsuspend and approve this store?')"
                                                    class="w-full bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition text-sm font-medium">
                                                Unsuspend
                                            </button>
                                        </form>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </li>
                @endforeach
            </ul>

            <!-- Pagination -->
            <div class="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
                {{ $stores->links() }}
            </div>
        @else
            <div class="px-4 py-12 text-center">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/>
                </svg>
                <h3 class="mt-2 text-sm font-medium text-gray-900">No stores found</h3>
                <p class="mt-1 text-sm text-gray-500">Try adjusting your filters or search terms.</p>
            </div>
        @endif
    </div>
</div>

<!-- Reject Modal -->
<div id="rejectModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 mb-4">Reject Store</h3>
            <form id="rejectForm" action="{{ route('admin.manual-approval.reject-store') }}" method="POST">
                @csrf
                <input type="hidden" name="store_id" id="reject_store_id">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Rejection Reason <span class="text-red-500">*</span></label>
                    <textarea name="rejection_reason" id="rejection_reason" rows="4" required
                              class="w-full rounded-md border-gray-300 shadow-sm focus:border-primary focus:ring-primary"
                              placeholder="Please provide a reason for rejecting this store..."></textarea>
                </div>

                <div class="flex gap-3">
                    <button type="submit" class="flex-1 bg-red-600 text-white px-4 py-2 rounded-md hover:bg-red-700 transition font-medium">
                        Reject Store
                    </button>
                    <button type="button" onclick="closeRejectModal()" 
                            class="flex-1 bg-gray-200 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-300 transition font-medium">
                        Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Suspend Modal -->
<div id="suspendModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <h3 class="text-lg font-medium text-gray-900 mb-4">Suspend Store</h3>
            <form id="suspendForm" action="{{ route('admin.manual-approval.suspend-store') }}" method="POST">
                @csrf
                <input type="hidden" name="store_id" id="suspend_store_id">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Suspension Reason (Optional)</label>
                    <textarea name="rejection_reason" id="suspension_reason" rows="4"
                              class="w-full rounded-md border-gray-300 shadow-sm focus:border-primary focus:ring-primary"
                              placeholder="Please provide a reason for suspending this store..."></textarea>
                </div>

                <div class="flex gap-3">
                    <button type="submit" class="flex-1 bg-yellow-600 text-white px-4 py-2 rounded-md hover:bg-yellow-700 transition font-medium">
                        Suspend Store
                    </button>
                    <button type="button" onclick="closeSuspendModal()" 
                            class="flex-1 bg-gray-200 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-300 transition font-medium">
                        Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showRejectModal(storeId, storeName) {
    document.getElementById('reject_store_id').value = storeId;
    document.getElementById('rejectModal').classList.remove('hidden');
}

function closeRejectModal() {
    document.getElementById('rejectModal').classList.add('hidden');
    document.getElementById('rejection_reason').value = '';
}

function showSuspendModal(storeId, storeName) {
    document.getElementById('suspend_store_id').value = storeId;
    document.getElementById('suspendModal').classList.remove('hidden');
}

function closeSuspendModal() {
    document.getElementById('suspendModal').classList.add('hidden');
    document.getElementById('suspension_reason').value = '';
}

// Close modals when clicking outside
window.onclick = function(event) {
    const rejectModal = document.getElementById('rejectModal');
    const suspendModal = document.getElementById('suspendModal');
    if (event.target === rejectModal) {
        closeRejectModal();
    }
    if (event.target === suspendModal) {
        closeSuspendModal();
    }
}
</script>

@push('styles')
<style>
    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 0.5rem;
    }
    .pagination a, .pagination span {
        padding: 0.5rem 1rem;
        border: 1px solid #d1d5db;
        border-radius: 0.375rem;
        color: #374151;
        text-decoration: none;
    }
    .pagination a:hover {
        background-color: #f3f4f6;
    }
    .pagination .active span {
        background-color: #ff5349;
        color: white;
        border-color: #ff5349;
    }
</style>
@endpush
@endsection
