@extends('layouts.app')

@section('title', 'Admin Dashboard')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="mb-8 flex items-center justify-between">
        <div>
            <h1 class="text-3xl font-bold text-gray-900">Admin Dashboard</h1>
            <p class="text-gray-600 mt-2">Welcome to the admin panel</p>
        </div>
        
        <div class="flex items-center gap-4">
            <!-- Download QR Code Link -->
            <a href="{{ route('admin.download-qrcode') }}" 
               class="inline-flex items-center gap-2 px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary-dark transition text-sm font-medium">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                </svg>
                Download QR Code
            </a>
            
            <!-- Notification Toggle -->
            <div class="bg-white rounded-lg shadow-md p-4 border border-gray-200">
                <div class="flex items-center space-x-4">
                    <div class="flex flex-col">
                        <label class="text-sm font-medium text-gray-700 mb-1">Browser Notifications</label>
                        <div class="flex items-center space-x-3">
                            <label class="relative inline-flex items-center cursor-pointer">
                                <input type="checkbox" id="browser-notifications-toggle" class="sr-only peer">
                                <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                            </label>
                            <span id="notification-status" class="text-xs text-gray-500">Off</span>
                        </div>
                    </div>
                    <div class="flex flex-col">
                        <label class="text-sm font-medium text-gray-700 mb-1">Sound</label>
                        <div class="flex items-center space-x-3">
                            <label class="relative inline-flex items-center cursor-pointer">
                                <input type="checkbox" id="sound-toggle" class="sr-only peer" checked>
                                <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                            </label>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    @if(session('success'))
        <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 mb-6" role="alert">
            <p>{{ session('success') }}</p>
        </div>
    @endif

    <!-- Revenue Statistics -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-gradient-to-br from-green-500 to-green-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-green-100">Total Revenue</p>
                    <p class="text-3xl font-bold mt-2">{{ currency($totalRevenue) }}</p>
                    <p class="text-xs text-green-100 mt-1">All orders ({{ currency($paidRevenue) }} paid)</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-blue-500 to-blue-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-blue-100">This Month</p>
                    <p class="text-3xl font-bold mt-2">{{ currency($thisMonthRevenue) }}</p>
                    <p class="text-xs text-blue-100 mt-1">All orders ({{ currency($thisMonthPaidRevenue) }} paid)</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-yellow-500 to-yellow-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-yellow-100">Today</p>
                    <p class="text-3xl font-bold mt-2">{{ currency($todayRevenue) }}</p>
                    <p class="text-xs text-yellow-100 mt-1">All orders ({{ currency($todayPaidRevenue ?? 0) }} paid)</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-gradient-to-br from-purple-500 to-purple-600 shadow-lg rounded-lg p-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-purple-100">Unpaid Revenue</p>
                    <p class="text-3xl font-bold mt-2">{{ currency($pendingRevenue) }}</p>
                    <p class="text-xs text-purple-100 mt-1">Orders not yet paid</p>
                </div>
                <div class="bg-white/20 p-3 rounded-full">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-600">Total Orders</p>
                    <p class="text-2xl font-bold text-gray-900 mt-2">{{ $stats['orders'] }}</p>
                    <p class="text-xs text-gray-500 mt-1">{{ $pendingOrders }} pending</p>
                </div>
                <div class="bg-primary/10 p-3 rounded-full">
                    <svg class="w-6 h-6 text-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-600">Total Products</p>
                    <p class="text-2xl font-bold text-gray-900 mt-2">{{ $stats['products'] }}</p>
                    <p class="text-xs text-gray-500 mt-1">{{ $stats['active_products'] }} active</p>
                </div>
                <div class="bg-primary/10 p-3 rounded-full">
                    <svg class="w-6 h-6 text-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-600">Total Users</p>
                    <p class="text-2xl font-bold text-gray-900 mt-2">{{ $stats['users'] }}</p>
                    <p class="text-xs text-gray-500 mt-1">Registered users</p>
                </div>
                <div class="bg-primary/10 p-3 rounded-full">
                    <svg class="w-6 h-6 text-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"/>
                    </svg>
                </div>
            </div>
        </div>

        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-sm font-medium text-gray-600">Order Status</p>
                    <p class="text-2xl font-bold text-gray-900 mt-2">{{ $completedOrders }}</p>
                    <p class="text-xs text-gray-500 mt-1">{{ $processingOrders }} processing</p>
                </div>
                <div class="bg-primary/10 p-3 rounded-full">
                    <svg class="w-6 h-6 text-primary" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        <!-- Recent Orders -->
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between mb-4">
                <h2 class="text-xl font-bold text-gray-900">Recent Orders</h2>
                <a href="{{ route('admin.orders.index') }}" class="text-primary hover:text-primary-dark text-sm font-medium">View All</a>
            </div>
            <div class="space-y-4">
                @forelse($recentOrders as $order)
                    <div class="border-b border-gray-200 pb-4 last:border-0 last:pb-0">
                        <div class="flex items-center justify-between">
                            <div class="flex-1">
                                <p class="font-semibold text-gray-900">{{ $order->order_number }}</p>
                                <p class="text-sm text-gray-600">{{ $order->user ? $order->user->name : $order->shipping_name }}</p>
                                <p class="text-xs text-gray-500 mt-1">{{ $order->created_at->format('M d, Y h:i A') }}</p>
                            </div>
                            <div class="text-right">
                                <p class="font-bold text-primary">{{ currency($order->total) }}</p>
                                <span class="inline-block px-2 py-1 text-xs rounded-full 
                                    {{ $order->status === 'completed' ? 'bg-green-100 text-green-800' : 
                                       ($order->status === 'pending' ? 'bg-yellow-100 text-yellow-800' : 
                                       ($order->status === 'processing' ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-800')) }}">
                                    {{ ucfirst($order->status) }}
                                </span>
                            </div>
                        </div>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No orders yet</p>
                @endforelse
            </div>
        </div>

        <!-- Top Selling Products -->
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center justify-between mb-4">
                <h2 class="text-xl font-bold text-gray-900">Top Selling Products</h2>
                <a href="{{ route('admin.products.index') }}" class="text-primary hover:text-primary-dark text-sm font-medium">View All</a>
            </div>
            <div class="space-y-4">
                @forelse($topProducts as $product)
                    <div class="border-b border-gray-200 pb-4 last:border-0 last:pb-0">
                        <div class="flex items-center justify-between">
                            <div class="flex-1">
                                <p class="font-semibold text-gray-900">{{ $product->product_name }}</p>
                                <p class="text-sm text-gray-600">{{ $product->total_sold }} sold</p>
                            </div>
                            <div class="text-right">
                                <p class="font-bold text-primary">{{ currency($product->total_revenue) }}</p>
                            </div>
                        </div>
                    </div>
                @empty
                    <p class="text-gray-500 text-center py-4">No sales data yet</p>
                @endforelse
            </div>
        </div>
    </div>

    <!-- Sales Chart (Last 7 Days) -->
    <div class="bg-white shadow rounded-lg p-6 mb-8">
        <h2 class="text-xl font-bold text-gray-900 mb-6">Sales Overview (Last 7 Days)</h2>
        <div class="h-64 flex items-end justify-between gap-2">
            @foreach($salesData as $data)
                <div class="flex-1 flex flex-col items-center">
                    <div class="w-full bg-gray-200 rounded-t relative" style="height: {{ $data['revenue'] > 0 ? max(20, ($data['revenue'] / max(array_column($salesData, 'revenue'))) * 100) : 0 }}%">
                        <div class="absolute inset-x-0 bottom-full mb-1 text-xs text-gray-600 text-center">
                            {{ currency($data['revenue']) }}
                        </div>
                    </div>
                    <p class="text-xs text-gray-500 mt-2">{{ $data['date'] }}</p>
                    <p class="text-xs text-gray-400">{{ $data['orders'] }} orders</p>
                </div>
            @endforeach
        </div>
    </div>

    <!-- Email Marketing Quick Access -->
    <div class="bg-gradient-to-br from-purple-500 to-purple-600 shadow-lg rounded-lg p-6 text-white">
        <div class="flex items-center justify-between">
            <div>
                <h2 class="text-2xl font-bold mb-2">Email Marketing</h2>
                <p class="text-purple-100 mb-4">Manage subscribers and send marketing emails to your audience</p>
                <a href="{{ route('admin.email-marketing.index') }}" 
                   class="inline-flex items-center gap-2 bg-white text-purple-600 px-6 py-3 rounded-lg font-bold hover:bg-purple-50 transition shadow-lg hover:shadow-xl">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                    </svg>
                    Go to Email Marketing
                </a>
            </div>
            <div class="bg-white/20 p-4 rounded-full">
                <svg class="w-12 h-12" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                </svg>
            </div>
        </div>
    </div>
</div>

<!-- Notification Sound will be generated using Web Audio API -->

<script>
(function() {
    'use strict';
    
    const notificationToggle = document.getElementById('browser-notifications-toggle');
    const soundToggle = document.getElementById('sound-toggle');
    const notificationStatus = document.getElementById('notification-status');
    
    let notificationCheckInterval = null;
    let preferences = {
        browser_notifications_enabled: false,
        sound_enabled: true,
        new_order_notifications: true,
        new_user_notifications: true,
    };
    
    // Load preferences on page load
    async function loadPreferences() {
        try {
            const response = await fetch('{{ route("admin.notifications.preferences") }}');
            const data = await response.json();
            if (data.success) {
                preferences = data.preferences;
                notificationToggle.checked = preferences.browser_notifications_enabled;
                soundToggle.checked = preferences.sound_enabled;
                updateStatus();
            }
        } catch (error) {
            console.error('Error loading preferences:', error);
        }
    }
    
    // Play notification sound using Web Audio API (plays on device)
    function playNotificationSound() {
        try {
            // Create audio context (may need user interaction first)
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            
            // Create a more noticeable notification sound (two-tone beep)
            const oscillator1 = audioContext.createOscillator();
            const oscillator2 = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            const masterGain = audioContext.createGain();
            
            oscillator1.connect(gainNode);
            oscillator2.connect(gainNode);
            gainNode.connect(masterGain);
            masterGain.connect(audioContext.destination);
            
            // First beep (higher pitch)
            oscillator1.frequency.value = 1000;
            oscillator1.type = 'sine';
            
            // Second beep (lower pitch) - starts after first
            oscillator2.frequency.value = 800;
            oscillator2.type = 'sine';
            
            // Set volume
            masterGain.gain.setValueAtTime(0.5, audioContext.currentTime);
            
            // First beep
            gainNode.gain.setValueAtTime(0.5, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.15);
            
            oscillator1.start(audioContext.currentTime);
            oscillator1.stop(audioContext.currentTime + 0.15);
            
            // Second beep (after a short pause)
            gainNode.gain.setValueAtTime(0.5, audioContext.currentTime + 0.2);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.35);
            
            oscillator2.start(audioContext.currentTime + 0.2);
            oscillator2.stop(audioContext.currentTime + 0.35);
        } catch (error) {
            console.log('Could not play notification sound:', error);
            // Fallback: try using HTML5 audio
            try {
                const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxuxuTemnFFBSuBzvLYiTcIGWi77eefTRAMbsbk3ppxRQU');
                audio.volume = 0.5;
                audio.play().catch(e => console.log('Audio fallback failed:', e));
            } catch (e) {
                console.log('All audio methods failed');
            }
        }
    }
    
    // Update status text
    function updateStatus() {
        if (preferences.browser_notifications_enabled) {
            notificationStatus.textContent = 'On';
            notificationStatus.classList.remove('text-gray-500');
            notificationStatus.classList.add('text-green-600', 'font-semibold');
        } else {
            notificationStatus.textContent = 'Off';
            notificationStatus.classList.remove('text-green-600', 'font-semibold');
            notificationStatus.classList.add('text-gray-500');
        }
    }
    
    // Request notification permission (required for system-level notifications)
    async function requestPermission() {
        if (!('Notification' in window)) {
            alert('This browser does not support system notifications. Please use a modern browser like Chrome, Firefox, or Edge.');
            return false;
        }
        
        if (Notification.permission === 'granted') {
            return true;
        }
        
        if (Notification.permission !== 'denied') {
            const permission = await Notification.requestPermission();
            if (permission === 'granted') {
                // Test notification to show it works
                const testNotification = new Notification('Notifications Enabled!', {
                    body: 'You will now receive notifications for new orders and customer signups.',
                    icon: '/favicon.ico',
                    tag: 'test-notification',
                });
                setTimeout(() => testNotification.close(), 3000);
                return true;
            }
            return false;
        }
        
        alert('Notification permission was denied. Please enable it in your browser settings:\n\nChrome/Edge: Settings > Privacy > Site Settings > Notifications\nFirefox: Preferences > Privacy & Security > Permissions > Notifications');
        return false;
    }
    
    // Show browser notification (system-level notification that appears on device)
    function showNotification(title, message, icon, url) {
        if (!preferences.browser_notifications_enabled) return;
        
        // Play sound if enabled (play before showing notification)
        if (preferences.sound_enabled) {
            playNotificationSound();
        }
        
        // Create system-level notification that appears on device
        const notification = new Notification(title, {
            body: message,
            icon: icon || '/favicon.ico',
            badge: '/favicon.ico',
            tag: 'admin-notification',
            requireInteraction: false,
            silent: false, // Ensure sound is not silenced
        });
        
        // Handle notification click
        notification.onclick = function(event) {
            event.preventDefault();
            window.focus();
            if (url) {
                window.location.href = url;
            }
            notification.close();
        };
        
        // Auto close after 8 seconds (longer for device notifications)
        setTimeout(() => {
            notification.close();
        }, 8000);
        
        // Also play sound when notification is shown (some browsers need this)
        notification.onshow = function() {
            if (preferences.sound_enabled) {
                // Small delay to ensure sound plays with notification
                setTimeout(() => {
                    playNotificationSound();
                }, 100);
            }
        };
    }
    
    // Check for new notifications
    async function checkNotifications() {
        if (!preferences.browser_notifications_enabled) return;
        
        try {
            const response = await fetch('{{ route("admin.notifications.check") }}');
            const data = await response.json();
            
            if (data.success && data.notifications && data.notifications.length > 0) {
                data.notifications.forEach(notif => {
                    showNotification(
                        notif.title,
                        notif.message,
                        notif.icon,
                        notif.url
                    );
                });
            }
        } catch (error) {
            console.error('Error checking notifications:', error);
        }
    }
    
    // Toggle browser notifications
    notificationToggle.addEventListener('change', async function() {
        const enabled = this.checked;
        
        if (enabled) {
            const hasPermission = await requestPermission();
            if (!hasPermission) {
                this.checked = false;
                return;
            }
        }
        
        try {
            const response = await fetch('{{ route("admin.notifications.toggle") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                },
                body: JSON.stringify({
                    field: 'browser_notifications_enabled',
                    value: enabled,
                }),
            });
            
            const data = await response.json();
            if (data.success) {
                preferences.browser_notifications_enabled = enabled;
                updateStatus();
                
                if (enabled) {
                    // Start checking for notifications every 10 seconds
                    if (notificationCheckInterval) {
                        clearInterval(notificationCheckInterval);
                    }
                    notificationCheckInterval = setInterval(checkNotifications, 10000);
                    // Check immediately
                    checkNotifications();
                } else {
                    // Stop checking
                    if (notificationCheckInterval) {
                        clearInterval(notificationCheckInterval);
                        notificationCheckInterval = null;
                    }
                }
            }
        } catch (error) {
            console.error('Error toggling notifications:', error);
            this.checked = !enabled;
        }
    });
    
    // Toggle sound
    soundToggle.addEventListener('change', async function() {
        const enabled = this.checked;
        
        try {
            const response = await fetch('{{ route("admin.notifications.toggle") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                },
                body: JSON.stringify({
                    field: 'sound_enabled',
                    value: enabled,
                }),
            });
            
            const data = await response.json();
            if (data.success) {
                preferences.sound_enabled = enabled;
            }
        } catch (error) {
            console.error('Error toggling sound:', error);
            this.checked = !enabled;
        }
    });
    
    // Initialize
    loadPreferences();
    
    // Check notifications on page load if enabled
    setTimeout(() => {
        if (preferences.browser_notifications_enabled) {
            checkNotifications();
            notificationCheckInterval = setInterval(checkNotifications, 10000);
        }
    }, 2000);
})();
</script>
@endsection
