<?php
/**
 * Router for subdirectory installation
 * Simplified and more robust version
 */

// Get the URI and base path
$uri = $_SERVER['REQUEST_URI'] ?? '/';
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';

// Extract base path from SCRIPT_NAME
// SCRIPT_NAME will be like "/Eurasian Express Limited/index.php" or "/Eurasian%20Express%20Limited/index.php"
$basePath = '';
if ($scriptName && strpos($scriptName, 'index.php') !== false) {
    // Get directory name (the subdirectory path)
    $basePath = dirname($scriptName);
    // Normalize: remove backslashes, handle special cases
    $basePath = str_replace('\\', '/', $basePath);
    // Remove leading ./ if present
    $basePath = ltrim($basePath, './');
    // Handle edge cases
    if ($basePath === '/' || $basePath === '.' || $basePath === '') {
        $basePath = '';
    } else {
        // Ensure it starts with /
        if ($basePath[0] !== '/') {
            $basePath = '/' . $basePath;
        }
    }
}

// Parse the URI
$parsedUri = parse_url($uri, PHP_URL_PATH);
$queryString = parse_url($uri, PHP_URL_QUERY);

// Fallback: If base path is empty but URI starts with a subdirectory, detect it from URI
// This handles cases where SCRIPT_NAME doesn't contain the subdirectory path
if (empty($basePath) && !empty($parsedUri) && $parsedUri !== '/') {
    $parts = explode('/', trim($parsedUri, '/'));
    if (!empty($parts[0])) {
        // Check if first part matches a directory name in the project root
        // This is a heuristic - if the directory exists, it's likely the subdirectory
        $possibleBasePath = '/' . $parts[0];
        $possibleDir = __DIR__ . str_replace('/', DIRECTORY_SEPARATOR, $possibleBasePath);
        // Only use this if the directory exists AND it's not a common Laravel route
        $commonRoutes = ['api', 'admin', 'login', 'register', 'logout', 'password', 'storage', 'css', 'js', 'images', 'products', 'cart', 'checkout'];
        if (is_dir($possibleDir) && !in_array(strtolower($parts[0]), $commonRoutes)) {
            $basePath = $possibleBasePath;
        }
    }
}

// Decode everything for comparison (handles both encoded and non-encoded)
$parsedUriDecoded = urldecode($parsedUri);
$basePathDecoded = urldecode($basePath);

// Determine what path to pass to Laravel
$cleanUri = $parsedUriDecoded;

// If we have a base path, try to strip it from the URI
if ($basePathDecoded && $basePathDecoded !== '/') {
    // Normalize both for comparison (trim trailing slashes)
    $basePathNormalized = rtrim($basePathDecoded, '/');
    $uriNormalized = $parsedUriDecoded;
    
    // Check if base path matches exactly (root of subdirectory)
    if ($uriNormalized === $basePathNormalized || $uriNormalized === $basePathNormalized . '/') {
        $cleanUri = '/';
    }
    // Check if base path is at the start of URI followed by /
    elseif (strpos($uriNormalized, $basePathNormalized . '/') === 0) {
        // Base path + something after it
        $cleanUri = substr($uriNormalized, strlen($basePathNormalized));
        // Ensure it starts with /
        if ($cleanUri === '' || $cleanUri[0] !== '/') {
            $cleanUri = '/' . $cleanUri;
        }
    }
    // Check if URI starts with base path directly (no trailing /)
    elseif (strpos($uriNormalized, $basePathNormalized) === 0 && strlen($uriNormalized) > strlen($basePathNormalized)) {
        $cleanUri = substr($uriNormalized, strlen($basePathNormalized));
        if ($cleanUri === '' || $cleanUri[0] !== '/') {
            $cleanUri = '/' . $cleanUri;
        }
    }
    else {
        // Base path not found at start - URI might already be clean
        // Check if URI doesn't start with base path at all
        $cleanUri = $parsedUriDecoded;
    }
}

// Ensure clean URI starts with /
if ($cleanUri === '' || $cleanUri[0] !== '/') {
    $cleanUri = '/' . $cleanUri;
}

// Serve static files from /public
// Check both cleaned URI and original URI (in case base path wasn't stripped correctly)
$staticUris = [$cleanUri];
if ($parsedUriDecoded !== $cleanUri) {
    $staticUris[] = $parsedUriDecoded;
}

foreach ($staticUris as $staticUri) {
    if ($staticUri !== '/' && preg_match('/\.(css|js|jpg|jpeg|png|gif|svg|ico|woff|woff2|ttf|eot|webp)$/i', $staticUri)) {
        // Try multiple path variations
        $possiblePaths = [
            __DIR__ . '/public' . $staticUri,
            __DIR__ . '/public' . str_replace($basePathDecoded, '', $staticUri),
        ];
        
        foreach ($possiblePaths as $file) {
            // Normalize Windows path separators
            $file = str_replace('/', DIRECTORY_SEPARATOR, $file);
            
            if (is_file($file) && file_exists($file)) {
                $mimeType = mime_content_type($file) ?: 'application/octet-stream';
                header('Content-Type: ' . $mimeType);
                header('Content-Length: ' . filesize($file));
                readfile($file);
                exit;
            }
        }
    }
}

// Update REQUEST_URI to the clean path for Laravel routing
// This is what Laravel's Request::capture() will read
if ($queryString) {
    $_SERVER['REQUEST_URI'] = $cleanUri . '?' . $queryString;
} else {
    $_SERVER['REQUEST_URI'] = $cleanUri;
}

// Also update PATH_INFO if it exists (some servers use this)
if (isset($_SERVER['PATH_INFO'])) {
    $_SERVER['PATH_INFO'] = $cleanUri;
}

// Preserve the base path in SCRIPT_NAME for RouteServiceProvider to detect it
$_SERVER['SCRIPT_FILENAME'] = __DIR__ . '/public/index.php';
// Ensure SCRIPT_NAME has the base path correctly formatted
if ($basePath) {
    $_SERVER['SCRIPT_NAME'] = $basePath . '/public/index.php';
} else {
    $_SERVER['SCRIPT_NAME'] = '/public/index.php';
}


// Change to public directory and load Laravel
chdir(__DIR__ . '/public');
require __DIR__ . '/public/index.php';
