<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;

class CurrencyService
{
    private $baseCurrency = 'GHS'; // Products are stored in GHS
    private $supportedCurrencies = ['USD', 'GHS', 'NGN'];
    
    /**
     * Get exchange rates from API (using exchangerate.host which provides Google-like rates)
     * Cache for 2 minutes to ensure rates stay current and update frequently
     */
    public function getExchangeRates()
    {
        return Cache::remember('exchange_rates', 120, function () { // 2 minutes cache for frequent updates
            // Try exchangerate.host first (free, accurate rates similar to Google)
            // Fetch rates with GHS as base, then convert to USD and NGN
            try {
                $response = Http::withOptions(['verify' => false])->timeout(5)->get('https://api.exchangerate.host/latest?base=GHS&symbols=USD,NGN');
                
                if ($response->successful()) {
                    $data = $response->json();
                    
                    if (isset($data['success']) && $data['success'] === true && isset($data['rates'])) {
                        $usdRate = (float) ($data['rates']['USD'] ?? 0);
                        $ngnRate = (float) ($data['rates']['NGN'] ?? 0);
                        
                        if ($usdRate > 0 && $ngnRate > 0) {
                            return [
                                'GHS' => 1.0,
                                'USD' => $usdRate,
                                'NGN' => $ngnRate,
                                'updated_at' => now()
                            ];
                        }
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Currency API Error (exchangerate.host): ' . $e->getMessage());
            }
            
            // Try exchangerate-api.com as fallback (fetch USD rates and convert)
            try {
                $response = Http::withOptions(['verify' => false])->timeout(5)->get('https://api.exchangerate-api.com/v4/latest/USD');
                
                if ($response->successful()) {
                    $data = $response->json();
                    
                    if (isset($data['rates']) && isset($data['rates']['GHS']) && isset($data['rates']['NGN'])) {
                        $ghsRate = (float) $data['rates']['GHS'];
                        $ngnRate = (float) $data['rates']['NGN'];
                        
                        // Convert: 1 GHS = 1/USD_to_GHS USD, 1 GHS = NGN/USD_to_GHS NGN
                        return [
                            'GHS' => 1.0,
                            'USD' => 1.0 / $ghsRate,
                            'NGN' => $ngnRate / $ghsRate,
                            'updated_at' => now()
                        ];
                    }
                }
            } catch (\Exception $e) {
                \Log::error('Currency API Error (exchangerate-api): ' . $e->getMessage());
            }
            
            // Fallback rates if all APIs fail (approximate rates - should be updated manually)
            // Based on current Google rates: 1 GHS ≈ 0.091 USD, 1 GHS ≈ 137 NGN
            return [
                'GHS' => 1.0,
                'USD' => 0.091,
                'NGN' => 137,
                'updated_at' => now()
            ];
        });
    }
    
    /**
     * Convert price from GHS to target currency
     * Returns rounded to nearest whole number
     */
    public function convert($amount, $toCurrency = 'GHS')
    {
        if (!in_array($toCurrency, $this->supportedCurrencies)) {
            $toCurrency = 'GHS';
        }
        
        $rates = $this->getExchangeRates();
        $rate = $rates[$toCurrency] ?? 1.0;
        
        $converted = $amount * $rate;
        
        // Round to nearest whole number
        return round($converted);
    }
    
    /**
     * Format price with currency symbol
     * Always shows whole numbers (no decimals)
     */
    public function format($amount, $currency = 'GHS')
    {
        // Ensure currency is valid, default to GHS if not
        if (!in_array($currency, $this->supportedCurrencies)) {
            $currency = 'GHS';
        }
        
        $converted = $this->convert($amount, $currency);
        
        $symbols = [
            'USD' => '$',
            'GHS' => 'GHS ',
            'NGN' => '₦'
        ];
        
        // Default to GHS symbol if currency not recognized (not dollar)
        $symbol = $symbols[$currency] ?? 'GHS ';
        
        // Always format as whole number (no decimals)
        return $symbol . number_format($converted, 0);
    }
    
    /**
     * Get current currency from session
     */
    public function getCurrentCurrency()
    {
        return session('currency', 'GHS'); // Default to GHS
    }
    
    /**
     * Set current currency
     */
    public function setCurrentCurrency($currency)
    {
        if (in_array($currency, $this->supportedCurrencies)) {
            session(['currency' => $currency]);
            return true;
        }
        return false;
    }
    
    /**
     * Get supported currencies
     */
    public function getSupportedCurrencies()
    {
        return $this->supportedCurrencies;
    }
    
    /**
     * Force refresh exchange rates (clear cache and fetch fresh rates)
     */
    public function refreshRates()
    {
        Cache::forget('exchange_rates');
        return $this->getExchangeRates();
    }
}

