<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class StoreProductController extends Controller
{
    /**
     * Display a listing of the store's products.
     */
    public function index()
    {
        $store = Auth::user()->store;
        $products = Product::where('store_id', $store->id)
            ->with(['category', 'categories'])
            ->latest()
            ->paginate(20);
        
        return view('store.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new product.
     */
    public function create()
    {
        $categories = Category::where('is_active', true)->orderBy('name')->get();
        return view('store.products.create', compact('categories'));
    }

    /**
     * Store a newly created product.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'categories' => 'required|array|min:1',
                'categories.*' => 'exists:categories,id',
                'name' => 'required|string|max:255',
                'description' => 'required|string',
                'short_description' => 'nullable|string|max:500',
                'price' => 'required|numeric|min:0',
                'compare_price' => 'nullable|numeric|min:0',
                'sku' => 'nullable|string|max:255|unique:products,sku',
                'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
                'images' => 'nullable|array',
                'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:5120',
                'size' => 'nullable|string|max:255',
                'color' => 'nullable|string|max:255',
                'brand' => 'nullable|string|max:255',
                'stock' => 'required|integer|min:0',
                'is_featured' => 'boolean',
                'is_active' => 'boolean',
                'on_sale' => 'boolean',
            ]);

            $store = Auth::user()->store;

            // Generate slug from name
            $validated['slug'] = Str::slug($validated['name']);
            
            // Ensure slug is unique
            $originalSlug = $validated['slug'];
            $counter = 1;
            while (Product::where('slug', $validated['slug'])->exists()) {
                $validated['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }

            // Handle main image upload
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('products', 'public');
                $validated['image'] = $imagePath;
            }

            // Handle additional images upload
            if ($request->hasFile('images')) {
                $imagePaths = [];
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products', 'public');
                    $imagePaths[] = $path;
                }
                $validated['images'] = $imagePaths;
            }

            $validated['is_featured'] = $request->has('is_featured');
            $validated['is_active'] = $request->has('is_active');
            $validated['on_sale'] = $request->has('on_sale');
            
            // Assign store and set category_id
            $validated['store_id'] = $store->id;
            $validated['category_id'] = $validated['categories'][0] ?? null;
            
            // Remove categories from validated array before creating product
            $categories = $validated['categories'];
            unset($validated['categories']);

            $product = Product::create($validated);
            
            // Sync categories
            $product->categories()->sync($categories);

            return redirect()->route('store.products.index')
                ->with('success', 'Product created successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            Log::error('Store product creation failed: ' . $e->getMessage(), ['exception' => $e]);
            return redirect()->back()
                ->with('error', 'Failed to create product. Please try again.')
                ->withInput();
        }
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Product $product)
    {
        $store = Auth::user()->store;
        
        // Ensure store owns this product
        if ($product->store_id !== $store->id) {
            return redirect()->route('store.products.index')
                ->with('error', 'Unauthorized access.');
        }

        $categories = Category::where('is_active', true)->orderBy('name')->get();
        $product->load('categories');
        return view('store.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified product.
     */
    public function update(Request $request, Product $product)
    {
        try {
            $store = Auth::user()->store;
            
            // Ensure store owns this product
            if ($product->store_id !== $store->id) {
                return redirect()->route('store.products.index')
                    ->with('error', 'Unauthorized access.');
            }

            $validated = $request->validate([
                'categories' => 'required|array|min:1',
                'categories.*' => 'exists:categories,id',
                'name' => 'required|string|max:255',
                'description' => 'required|string',
                'short_description' => 'nullable|string|max:500',
                'price' => 'required|numeric|min:0',
                'compare_price' => 'nullable|numeric|min:0',
                'sku' => 'nullable|string|max:255|unique:products,sku,' . $product->id,
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
                'images' => 'nullable|array',
                'images.*' => 'image|mimes:jpeg,png,jpg,gif,webp|max:5120',
                'size' => 'nullable|string|max:255',
                'color' => 'nullable|string|max:255',
                'brand' => 'nullable|string|max:255',
                'stock' => 'required|integer|min:0',
                'is_featured' => 'boolean',
                'is_active' => 'boolean',
                'on_sale' => 'boolean',
            ]);

            // Update slug if name changed
            if ($validated['name'] !== $product->name) {
                $validated['slug'] = Str::slug($validated['name']);
                $originalSlug = $validated['slug'];
                $counter = 1;
                while (Product::where('slug', $validated['slug'])->where('id', '!=', $product->id)->exists()) {
                    $validated['slug'] = $originalSlug . '-' . $counter;
                    $counter++;
                }
            }

            // Handle main image upload
            if ($request->hasFile('image')) {
                // Delete old image
                if ($product->image) {
                    Storage::disk('public')->delete($product->image);
                }
                $imagePath = $request->file('image')->store('products', 'public');
                $validated['image'] = $imagePath;
            }

            // Handle additional images upload
            if ($request->hasFile('images')) {
                // Delete old additional images
                if ($product->images && is_array($product->images)) {
                    foreach ($product->images as $oldImage) {
                        Storage::disk('public')->delete($oldImage);
                    }
                }
                $imagePaths = [];
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products', 'public');
                    $imagePaths[] = $path;
                }
                $validated['images'] = $imagePaths;
            }

            $validated['is_featured'] = $request->has('is_featured');
            $validated['is_active'] = $request->has('is_active');
            $validated['on_sale'] = $request->has('on_sale');
            
            // Set category_id to first category for backward compatibility
            $validated['category_id'] = $validated['categories'][0] ?? null;
            
            // Remove categories from validated array before updating product
            $categories = $validated['categories'];
            unset($validated['categories']);

            $product->update($validated);
            
            // Sync categories
            $product->categories()->sync($categories);

            return redirect()->route('store.products.index')
                ->with('success', 'Product updated successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            Log::error('Store product update failed: ' . $e->getMessage(), ['exception' => $e]);
            return redirect()->back()
                ->with('error', 'Failed to update product. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove the specified product.
     */
    public function destroy(Product $product)
    {
        try {
            $store = Auth::user()->store;
            
            // Ensure store owns this product
            if ($product->store_id !== $store->id) {
                return redirect()->route('store.products.index')
                    ->with('error', 'Unauthorized access.');
            }

            // Delete main image
            if ($product->image) {
                Storage::disk('public')->delete($product->image);
            }

            // Delete additional images
            if ($product->images && is_array($product->images)) {
                foreach ($product->images as $image) {
                    Storage::disk('public')->delete($image);
                }
            }

            $product->delete();

            return redirect()->route('store.products.index')
                ->with('success', 'Product deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Store product deletion failed: ' . $e->getMessage(), ['exception' => $e]);
            return redirect()->route('store.products.index')
                ->with('error', 'Failed to delete product. Please try again.');
        }
    }
}

