<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;

use App\Models\Store;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class StoreController extends Controller
{
    /**
     * Show store registration form
     */
    public function showRegisterForm()
    {
        if (auth()->check() && auth()->user()->store) {
            return redirect()->route('store.dashboard');
        }
        return view('store.register');
    }

    /**
     * Register a new store
     */
    public function register(Request $request)
    {
        try {
            $user = null;

            if (auth()->check()) {
                $user = auth()->user();
                
                if ($user->store) {
                    return redirect()->route('store.dashboard')
                        ->with('info', 'You already have a store account.');
                }

                $validated = $request->validate([
                    'business_name' => 'required|string|max:255|unique:stores,business_name',
                    'phone' => 'nullable|string|max:20',
                    'description' => 'nullable|string',
                    'address' => 'nullable|string|max:255',
                    'city' => 'nullable|string|max:100',
                    'state' => 'nullable|string|max:100',
                    'country' => 'nullable|string|max:100',
                    'postal_code' => 'nullable|string|max:20',
                    'website' => 'nullable|url|max:255',
                    'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                ]);

                // Update user to be store owner
                $user->update(['is_store_owner' => true]);

                // Reload the store relationship in case it's just been created
                $user->refresh();
                // Explicitly attach the store relation for this user
                $user->setRelation('store', Store::where('user_id', $user->id)->latest()->first());

            } else {
                $validated = $request->validate([
                    'name' => 'required|string|max:255',
                    'email' => 'required|string|email|max:255|unique:users',
                    'password' => 'required|string|min:8|confirmed',
                    'business_name' => 'required|string|max:255|unique:stores,business_name',
                    'phone' => 'nullable|string|max:20',
                    'description' => 'nullable|string',
                    'address' => 'nullable|string|max:255',
                    'city' => 'nullable|string|max:100',
                    'state' => 'nullable|string|max:100',
                    'country' => 'nullable|string|max:100',
                    'postal_code' => 'nullable|string|max:20',
                    'website' => 'nullable|url|max:255',
                    'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                ]);

                // Create user account
                $user = User::create([
                    'name' => $validated['name'],
                    'email' => $validated['email'],
                    'password' => Hash::make($validated['password']),
                    'is_store_owner' => true,
                ]);
            }

            // Handle logo upload
            $logoPath = null;
            if ($request->hasFile('logo')) {
                try {
                    $logoPath = $request->file('logo')->store('stores/logos', 'public');
                } catch (\Exception $e) {
                    if (!auth()->check()) {
                        $user->delete();
                    }
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Failed to upload logo: ' . $e->getMessage());
                }
            }

            // Create store profile
            $store = Store::create([
                'user_id' => $user->id,
                'business_name' => $validated['business_name'],
                'slug' => Str::slug($validated['business_name']),
                'description' => $validated['description'] ?? null,
                'phone' => $validated['phone'] ?? null,
                'logo' => $logoPath,
                'address' => $validated['address'] ?? null,
                'city' => $validated['city'] ?? null,
                'state' => $validated['state'] ?? null,
                'country' => $validated['country'] ?? null,
                'postal_code' => $validated['postal_code'] ?? null,
                'website' => $validated['website'] ?? null,
                'status' => 'pending', // Requires admin approval
            ]);

            // Auto-login the user if not already logged in
            if (!auth()->check()) {
                auth()->login($user);
            }

            return redirect()->route('store.dashboard')
                ->with('success', 'Store account created successfully! Your account is pending admin approval.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to create store account: ' . $e->getMessage());
        }
    }

    /**
     * Show store dashboard
     */
    public function dashboard()
    {
        $store = auth()->user()->store;
        
        if (!$store) {
            return redirect()->route('store.register')
                ->with('error', 'Please complete your store registration.');
        }

        // Get store product IDs for order calculations
        $storeProductIds = $store->products()->pluck('id');
        
        // Get orders containing store products
        $orderIds = \App\Models\OrderItem::whereIn('product_id', $storeProductIds)
            ->distinct()
            ->pluck('order_id');
        
        $orders = \App\Models\Order::whereIn('id', $orderIds);
        
        // Statistics
        $stats = [
            'total_products' => $store->products()->count(),
            'active_products' => $store->products()->where('is_active', true)->count(),
            'inactive_products' => $store->products()->where('is_active', false)->count(),
            'total_orders' => $orders->count(),
            'pending_orders' => $orders->where('status', 'pending')->count(),
            'completed_orders' => $orders->where('status', 'completed')->count(),
            'total_sales' => $store->total_sales ?? 0,
            'total_earnings' => $store->total_earnings ?? 0,
            'pending_earnings' => $store->pending_earnings ?? 0,
            'total_customers' => $orders->distinct('user_id')->count('user_id'),
            'today_orders' => (clone $orders)->whereDate('created_at', today())->count(),
            'today_earnings' => (clone $orders)->whereDate('created_at', today())->where('status', 'completed')->sum('total') ?? 0,
            'this_month_orders' => (clone $orders)->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->count(),
            'this_month_earnings' => (clone $orders)->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->where('status', 'completed')->sum('total') ?? 0,
        ];

        // Recent orders (last 10)
        $recentOrders = $orders->with(['user', 'items.product'])
            ->latest()
            ->take(10)
            ->get();

        // Top selling products
        $topProducts = \App\Models\OrderItem::whereIn('product_id', $storeProductIds)
            ->selectRaw('product_id, product_name, SUM(quantity) as total_sold, SUM(subtotal) as total_revenue')
            ->groupBy('product_id', 'product_name')
            ->orderBy('total_sold', 'desc')
            ->take(5)
            ->get();

        // Sales chart data (last 7 days)
        $salesChartData = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $daySales = (clone $orders)->whereDate('created_at', $date)->where('status', 'completed')->sum('total') ?? 0;
            $dayOrders = (clone $orders)->whereDate('created_at', $date)->count();
            
            $salesChartData[] = [
                'date' => $date->format('M d'),
                'day' => $date->format('D'),
                'sales' => $daySales,
                'orders' => $dayOrders,
            ];
        }

        return view('store.dashboard', compact('store', 'stats', 'recentOrders', 'topProducts', 'salesChartData'));
    }

    /**
     * Show store profile page
     */
    public function show($slug)
    {
        $store = Store::where('slug', $slug)
            ->where('status', 'approved')
            ->firstOrFail();

        $products = $store->products()
            ->where('is_active', true)
            ->latest()
            ->paginate(12);

        return view('store.show', compact('store', 'products'));
    }
}


