<?php

namespace App\Http\Controllers\Store;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use App\Models\User;
use App\Models\StoreStaff;

class StaffController extends Controller
{
    public function index()
    {
        $store = Auth::user()->store;
        $staff = $store->staff()->with('user')->latest()->paginate(20);
        return view('store.staff.index', compact('store', 'staff'));
    }

    public function create()
    {
        return view('store.staff.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
            'role' => 'required|string|max:255',
        ]);

        $store = Auth::user()->store;
        $user = User::where('email', $request->email)->first();

        if ($store->staff()->where('user_id', $user->id)->exists()) {
            return back()->with('error', 'This user is already a staff member.');
        }

        $store->staff()->create([
            'user_id' => $user->id,
            'role' => $request->role,
        ]);

        return redirect()->route('store.staff.index')->with('success', 'Staff member invited successfully.');
    }

    public function edit(StoreStaff $staff)
    {
        $this->authorize('update', $staff);
        return view('store.staff.edit', compact('staff'));
    }

    public function update(Request $request, StoreStaff $staff)
    {
        $this->authorize('update', $staff);

        $request->validate([
            'role' => 'required|string|max:255',
        ]);

        $staff->update($request->only('role'));

        return redirect()->route('store.staff.index')->with('success', 'Staff member updated successfully.');
    }

    public function destroy(StoreStaff $staff)
    {
        $this->authorize('delete', $staff);
        $staff->delete();
        return redirect()->route('store.staff.index')->with('success', 'Staff member removed successfully.');
    }
}
