<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\Wishlist;
use App\Models\Address;
use App\Models\Product;
use Carbon\Carbon;

class CustomerDashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        // Statistics
        $totalOrders = Order::where('user_id', $user->id)->count();
        $pendingOrders = Order::where('user_id', $user->id)->where('status', 'pending')->count();
        $completedOrders = Order::where('user_id', $user->id)->where('status', 'completed')->count();
        $totalSpent = Order::where('user_id', $user->id)->where('payment_status', 'paid')->sum('total');
        $wishlistCount = Wishlist::where('user_id', $user->id)->count();
        $addressCount = Address::where('user_id', $user->id)->count();

        // Recent orders
        $recentOrders = Order::where('user_id', $user->id)
            ->latest()
            ->limit(5)
            ->get();

        // Wishlist items
        $wishlistItems = Wishlist::where('user_id', $user->id)
            ->with('product')
            ->latest()
            ->limit(4)
            ->get()
            ->filter(function ($item) {
                return $item->product !== null;
            });

        return view('customer.dashboard', compact(
            'totalOrders',
            'pendingOrders',
            'completedOrders',
            'totalSpent',
            'wishlistCount',
            'addressCount',
            'recentOrders',
            'wishlistItems'
        ));
    }

    public function wishlist()
    {
        $wishlistItems = Wishlist::where('user_id', Auth::id())
            ->with('product')
            ->latest()
            ->paginate(12);

        // Filter out items with deleted products
        $wishlistItems->getCollection()->transform(function ($item) {
            if ($item->product === null) {
                $item->delete();
                return null;
            }
            return $item;
        })->filter();

        return view('customer.wishlist', compact('wishlistItems'));
    }

    public function addToWishlist(Request $request, $productId)
    {
        $product = Product::findOrFail($productId);

        $wishlist = Wishlist::firstOrCreate([
            'user_id' => Auth::id(),
            'product_id' => $product->id,
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'added' => true,
                'message' => 'Product added to wishlist',
            ]);
        }

        return back()->with('success', 'Product added to wishlist!');
    }

    public function removeFromWishlist($productId)
    {
        Wishlist::where('user_id', Auth::id())
            ->where('product_id', $productId)
            ->delete();

        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'added' => false,
                'message' => 'Product removed from wishlist',
            ]);
        }

        return back()->with('success', 'Product removed from wishlist!');
    }

    public function addresses()
    {
        $addresses = Address::where('user_id', Auth::id())
            ->latest()
            ->get();

        return view('customer.addresses', compact('addresses'));
    }

    public function storeAddress(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:shipping,billing',
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address' => 'required|string',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
            'is_default' => 'nullable|boolean',
        ]);

        // If this is set as default, unset other defaults
        if ($request->has('is_default') && $request->is_default) {
            Address::where('user_id', Auth::id())
                ->where('type', $validated['type'])
                ->update(['is_default' => false]);
        }

        $address = Address::create([
            'user_id' => Auth::id(),
            'type' => $validated['type'],
            'full_name' => $validated['full_name'],
            'phone' => $validated['phone'],
            'address' => $validated['address'],
            'city' => $validated['city'] ?? null,
            'state' => $validated['state'] ?? null,
            'postal_code' => $validated['postal_code'] ?? null,
            'country' => $validated['country'] ?? null,
            'is_default' => $request->has('is_default') && $request->is_default,
        ]);

        return redirect()->route('customer.addresses')->with('success', 'Address added successfully!');
    }

    public function editAddress($id)
    {
        $address = Address::where('user_id', Auth::id())->findOrFail($id);
        return response()->json($address);
    }

    public function updateAddress(Request $request, $id)
    {
        $address = Address::where('user_id', Auth::id())->findOrFail($id);

        $validated = $request->validate([
            'type' => 'required|in:shipping,billing',
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address' => 'required|string',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
            'is_default' => 'nullable|boolean',
        ]);

        // If this is set as default, unset other defaults
        if ($request->has('is_default') && $request->is_default) {
            Address::where('user_id', Auth::id())
                ->where('type', $validated['type'])
                ->where('id', '!=', $id)
                ->update(['is_default' => false]);
        }

        $address->update($validated);

        return redirect()->route('customer.addresses')->with('success', 'Address updated successfully!');
    }

    public function deleteAddress($id)
    {
        $address = Address::where('user_id', Auth::id())->findOrFail($id);
        $address->delete();

        return redirect()->route('customer.addresses')->with('success', 'Address deleted successfully!');
    }

    public function setDefaultAddress($id)
    {
        $address = Address::where('user_id', Auth::id())->findOrFail($id);

        // Unset other defaults of the same type
        Address::where('user_id', Auth::id())
            ->where('type', $address->type)
            ->where('id', '!=', $id)
            ->update(['is_default' => false]);

        $address->update(['is_default' => true]);

        return back()->with('success', 'Default address updated!');
    }
}
