<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Models\User;
use App\Events\NewUserRegistered;
use App\Mail\WelcomeEmail;
use App\Mail\OtpEmail;
use App\Mail\PasswordResetEmail;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $user = Auth::user();
            
            // Check if email is verified
            if (!$user->email_verified_at) {
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                
                // Generate new OTP if user doesn't have one or it expired
                if (!$user->email_verification_otp || Carbon::now()->isAfter($user->email_verification_otp_expires_at)) {
                    // Generate secure 6-digit OTP (avoid sequential and repeated patterns)
                    do {
                        $otp = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
                    } while (
                        // Avoid sequential patterns (123456, 234567, etc.)
                        preg_match('/(012345|123456|234567|345678|456789|567890|098765|987654|876543|765432|654321|543210)/', $otp) ||
                        // Avoid repeated digits (000000, 111111, etc.)
                        preg_match('/(\d)\1{5}/', $otp) ||
                        // Avoid simple patterns (112233, 121212, etc.)
                        preg_match('/(\d)\1{2}(\d)\2{2}/', $otp) ||
                        preg_match('/(\d)(\d)\1\2\1\2/', $otp)
                    );
                    $otpExpiresAt = Carbon::now()->addMinutes(15);
                    
                    $user->update([
                        'email_verification_otp' => $otp,
                        'email_verification_otp_expires_at' => $otpExpiresAt,
                    ]);
                    
                    try {
                        Mail::to($user->email)->send(new OtpEmail($user, $otp));
                        \Log::info('OTP email sent successfully to: ' . $user->email);
                    } catch (\Exception $e) {
                        \Log::error('Failed to send OTP email to ' . $user->email . ': ' . $e->getMessage());
                        \Log::error('Email error trace: ' . $e->getTraceAsString());
                    }
                }
                
                $request->session()->put('verifying_user_id', $user->id);
                return redirect()->route('verify.email')->with('error', 'Please verify your email address before logging in. A verification code has been sent to your email.');
            }
            
            $request->session()->regenerate();
            
            // Redirect admins to dashboard, others to intended page or home
            if ($user->is_admin) {
                return redirect()->intended(route('admin.dashboard'));
            }
            
            return redirect()->intended('/');
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function showRegisterForm()
    {
        return view('auth.register');
    }

    public function register(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
        ]);

        // Generate secure 6-digit OTP (avoid sequential and repeated patterns)
        do {
            $otp = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (
            // Avoid sequential patterns (123456, 234567, etc.)
            preg_match('/(012345|123456|234567|345678|456789|567890|098765|987654|876543|765432|654321|543210)/', $otp) ||
            // Avoid repeated digits (000000, 111111, etc.)
            preg_match('/(\d)\1{5}/', $otp) ||
            // Avoid simple patterns (112233, 121212, etc.)
            preg_match('/(\d)\1{2}(\d)\2{2}/', $otp) ||
            preg_match('/(\d)(\d)\1\2\1\2/', $otp)
        );
        $otpExpiresAt = Carbon::now()->addMinutes(15);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => bcrypt($validated['password']),
            'email_verification_otp' => $otp,
            'email_verification_otp_expires_at' => $otpExpiresAt,
        ]);

        // Send OTP email
        try {
            Mail::to($user->email)->send(new OtpEmail($user, $otp));
            \Log::info('OTP email sent successfully to: ' . $user->email);
        } catch (\Exception $e) {
            // Log detailed error
            \Log::error('Failed to send OTP email to ' . $user->email . ': ' . $e->getMessage());
            \Log::error('Email error trace: ' . $e->getTraceAsString());
            
            // Still redirect but show a message that email might not have been sent
            return redirect()->route('verify.email')
                ->with('warning', 'Account created successfully! However, we encountered an issue sending the verification email. Please try resending the code.')
                ->with('user_id', $user->id);
        }

        // Store user ID in session for verification
        $request->session()->put('verifying_user_id', $user->id);
        
        // Store OTP in session temporarily for display if email fails (for debugging)
        $request->session()->put('last_otp', $otp);
        $request->session()->put('otp_expires_at', $otpExpiresAt->toDateTimeString());

        return redirect()->route('verify.email')->with('success', 'Account created successfully! Please check your email for the verification code.');
    }

    public function showVerifyEmailForm()
    {
        if (!session()->has('verifying_user_id')) {
            return redirect()->route('register')->with('error', 'Please register first.');
        }

        $user = User::find(session('verifying_user_id'));
        
        if (!$user) {
            session()->forget('verifying_user_id');
            return redirect()->route('register')->with('error', 'User not found. Please register again.');
        }

        // Refresh user to get latest OTP from database
        $user->refresh();
        
        // Get last OTP from session for debugging (if available)
        $lastOtp = session('last_otp', null);
        
        return view('auth.verify-email', compact('user', 'lastOtp'));
    }

    public function verifyEmail(Request $request)
    {
        $request->validate([
            'otp' => 'required|string',
        ]);

        if (!session()->has('verifying_user_id')) {
            return redirect()->route('register')->with('error', 'Please register first.');
        }

        $user = User::find(session('verifying_user_id'));
        
        if (!$user) {
            session()->forget('verifying_user_id');
            return redirect()->route('register')->with('error', 'User not found. Please register again.');
        }

        // Refresh user from database to ensure we have the latest OTP
        $user->refresh();

        // Normalize OTP input: trim whitespace, remove any non-numeric characters, ensure 6 digits
        $inputOtp = preg_replace('/[^0-9]/', '', trim($request->otp));
        
        // Ensure input is exactly 6 digits
        if (strlen($inputOtp) !== 6) {
            \Log::warning('OTP verification failed - invalid length', [
                'user_id' => $user->id,
                'input_length' => strlen($inputOtp),
                'raw_input' => $request->otp,
                'cleaned_input' => $inputOtp
            ]);
            return back()->withErrors(['otp' => 'Please enter a 6-digit verification code.'])->withInput();
        }

        // Get stored OTP and normalize it
        $storedOtp = (string) trim($user->email_verification_otp ?? '');
        
        // Check if stored OTP exists
        if (empty($storedOtp)) {
            \Log::warning('OTP verification failed - no stored OTP', [
                'user_id' => $user->id,
                'user_email' => $user->email
            ]);
            return back()->withErrors(['otp' => 'No verification code found. Please request a new one.'])->withInput();
        }
        
        // Remove any non-numeric characters from stored OTP (just in case)
        $storedOtp = preg_replace('/[^0-9]/', '', $storedOtp);
        
        // Pad stored OTP to 6 digits if needed (shouldn't be necessary, but just in case)
        $storedOtp = str_pad($storedOtp, 6, '0', STR_PAD_LEFT);

        // Compare as strings using strict comparison
        if ($storedOtp !== $inputOtp) {
            \Log::warning('OTP verification failed - mismatch', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'stored_otp' => $storedOtp,
                'stored_otp_length' => strlen($storedOtp),
                'stored_otp_type' => gettype($user->email_verification_otp),
                'stored_otp_raw' => $user->email_verification_otp,
                'input_otp' => $inputOtp,
                'input_otp_length' => strlen($inputOtp),
                'raw_input' => $request->otp,
                'are_equal' => ($storedOtp === $inputOtp),
                'strcmp_result' => strcmp($storedOtp, $inputOtp)
            ]);
            return back()->withErrors(['otp' => 'Invalid verification code. Please try again.'])->withInput();
        }

        // Check if OTP has expired
        if (Carbon::now()->isAfter($user->email_verification_otp_expires_at)) {
            return back()->withErrors(['otp' => 'Verification code has expired. Please request a new one.'])->withInput();
        }

        // Check if this is the first time verifying AND user was just created (within last 10 minutes)
        // This ensures welcome email is only sent during registration, not when logging in with unverified email
        $isFirstVerification = is_null($user->email_verified_at);
        $isNewRegistration = $isFirstVerification && $user->created_at && Carbon::now()->diffInMinutes($user->created_at) <= 10;

        // Verify email
        $user->update([
            'email_verified_at' => Carbon::now(),
            'email_verification_otp' => null,
            'email_verification_otp_expires_at' => null,
        ]);

        // Send welcome email ONLY if this is a new registration (user created within last 10 minutes)
        // NOT when user logs in with unverified email and then verifies
        if ($isNewRegistration) {
            try {
                Mail::to($user->email)->send(new WelcomeEmail($user));
            } catch (\Exception $e) {
                // Log error but don't break verification
                \Log::error('Failed to send welcome email: ' . $e->getMessage());
            }
            
            // Fire event for admin notifications (only for non-admin users)
            if (!$user->is_admin) {
                event(new NewUserRegistered($user));
            }
        }

        // Clear session
        $request->session()->regenerate();
        session()->forget('verifying_user_id');

        // Login user (with remember me for 2 weeks)
        Auth::login($user, true);

        $siteTitle = \App\Models\Setting::get('site_title', 'Vibrant Vogue');
        return redirect('/')->with('success', 'Email verified successfully! Welcome to ' . $siteTitle . '!');
    }

    public function resendOtp(Request $request)
    {
        if (!session()->has('verifying_user_id')) {
            return redirect()->route('register')->with('error', 'Please register first.');
        }

        $user = User::find(session('verifying_user_id'));
        
        if (!$user) {
            session()->forget('verifying_user_id');
            return redirect()->route('register')->with('error', 'User not found. Please register again.');
        }

        // Generate secure 6-digit OTP (avoid sequential and repeated patterns)
        do {
            $otp = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (
            // Avoid sequential patterns (123456, 234567, etc.)
            preg_match('/(012345|123456|234567|345678|456789|567890|098765|987654|876543|765432|654321|543210)/', $otp) ||
            // Avoid repeated digits (000000, 111111, etc.)
            preg_match('/(\d)\1{5}/', $otp) ||
            // Avoid simple patterns (112233, 121212, etc.)
            preg_match('/(\d)\1{2}(\d)\2{2}/', $otp) ||
            preg_match('/(\d)(\d)\1\2\1\2/', $otp)
        );
        $otpExpiresAt = Carbon::now()->addMinutes(15);

        $user->update([
            'email_verification_otp' => $otp,
            'email_verification_otp_expires_at' => $otpExpiresAt,
        ]);

        // Send OTP email
        try {
            Mail::to($user->email)->send(new OtpEmail($user, $otp));
            \Log::info('OTP email resent successfully to: ' . $user->email);
        } catch (\Exception $e) {
            // Log detailed error
            \Log::error('Failed to resend OTP email to ' . $user->email . ': ' . $e->getMessage());
            \Log::error('Email error trace: ' . $e->getTraceAsString());
            return back()->with('error', 'Failed to resend verification code. Please check your email settings or contact support.');
        }

        return back()->with('success', 'Verification code has been resent to your email.');
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/');
    }

    public function showForgotPasswordForm()
    {
        return view('auth.forgot-password');
    }

    public function sendPasswordResetOtp(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors(['email' => 'We could not find a user with that email address.'])->withInput();
        }

        // Generate secure 6-digit OTP (avoid sequential and repeated patterns)
        do {
            $otp = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (
            // Avoid sequential patterns (123456, 234567, etc.)
            preg_match('/(012345|123456|234567|345678|456789|567890|098765|987654|876543|765432|654321|543210)/', $otp) ||
            // Avoid repeated digits (000000, 111111, etc.)
            preg_match('/(\d)\1{5}/', $otp) ||
            // Avoid simple patterns (112233, 121212, etc.)
            preg_match('/(\d)\1{2}(\d)\2{2}/', $otp) ||
            preg_match('/(\d)(\d)\1\2\1\2/', $otp)
        );
        $otpExpiresAt = Carbon::now()->addMinutes(15);

        $user->update([
            'password_reset_otp' => $otp,
            'password_reset_otp_expires_at' => $otpExpiresAt,
        ]);

        // Send password reset OTP email
        try {
            Mail::to($user->email)->send(new PasswordResetEmail($user, $otp));
            \Log::info('Password reset OTP email sent successfully to: ' . $user->email);
        } catch (\Exception $e) {
            \Log::error('Failed to send password reset OTP email to ' . $user->email . ': ' . $e->getMessage());
            \Log::error('Email error trace: ' . $e->getTraceAsString());
            return back()->with('error', 'Failed to send password reset code. Please try again later.')->withInput();
        }

        // Store user ID in session for verification
        $request->session()->put('resetting_password_user_id', $user->id);
        $request->session()->put('last_password_reset_otp', $otp);
        $request->session()->put('password_reset_otp_expires_at', $otpExpiresAt->toDateTimeString());

        return redirect()->route('password.reset')->with('success', 'Password reset code has been sent to your email address.');
    }

    public function showResetPasswordForm()
    {
        if (!session()->has('resetting_password_user_id')) {
            return redirect()->route('password.forgot')->with('error', 'Please request a password reset first.');
        }

        $user = User::find(session('resetting_password_user_id'));
        
        if (!$user) {
            session()->forget('resetting_password_user_id');
            return redirect()->route('password.forgot')->with('error', 'User not found. Please request a new password reset.');
        }

        // Refresh user to get latest OTP from database
        $user->refresh();
        
        // Get last OTP from session for debugging (if available)
        $lastOtp = session('last_password_reset_otp', null);

        return view('auth.reset-password', compact('user', 'lastOtp'));
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'otp' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if (!session()->has('resetting_password_user_id')) {
            return redirect()->route('password.forgot')->with('error', 'Please request a password reset first.');
        }

        $user = User::find(session('resetting_password_user_id'));
        
        if (!$user) {
            session()->forget('resetting_password_user_id');
            return redirect()->route('password.forgot')->with('error', 'User not found. Please request a new password reset.');
        }

        // Refresh user from database to ensure we have the latest OTP
        $user->refresh();

        // Normalize OTP input: trim whitespace, remove any non-numeric characters, ensure 6 digits
        $inputOtp = preg_replace('/[^0-9]/', '', trim($request->otp));
        
        // Ensure input is exactly 6 digits
        if (strlen($inputOtp) !== 6) {
            return back()->withErrors(['otp' => 'Please enter a 6-digit verification code.'])->withInput();
        }

        // Get stored OTP and normalize it
        $storedOtp = (string) trim($user->password_reset_otp ?? '');
        
        // Check if stored OTP exists
        if (empty($storedOtp)) {
            return back()->withErrors(['otp' => 'No verification code found. Please request a new one.'])->withInput();
        }
        
        // Remove any non-numeric characters from stored OTP
        $storedOtp = preg_replace('/[^0-9]/', '', $storedOtp);
        
        // Pad stored OTP to 6 digits if needed
        $storedOtp = str_pad($storedOtp, 6, '0', STR_PAD_LEFT);

        // Compare as strings using strict comparison
        if ($storedOtp !== $inputOtp) {
            \Log::warning('Password reset OTP verification failed', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'stored_otp' => $storedOtp,
                'input_otp' => $inputOtp,
            ]);
            return back()->withErrors(['otp' => 'Invalid verification code. Please try again.'])->withInput();
        }

        // Check if OTP has expired
        if (Carbon::now()->isAfter($user->password_reset_otp_expires_at)) {
            return back()->withErrors(['otp' => 'Verification code has expired. Please request a new one.'])->withInput();
        }

        // Reset password
        $user->update([
            'password' => bcrypt($request->password),
            'password_reset_otp' => null,
            'password_reset_otp_expires_at' => null,
        ]);

        // Clear session
        $request->session()->forget('resetting_password_user_id');
        $request->session()->forget('last_password_reset_otp');
        $request->session()->forget('password_reset_otp_expires_at');

        return redirect()->route('login')->with('success', 'Your password has been reset successfully! You can now login with your new password.');
    }

    public function resendPasswordResetOtp(Request $request)
    {
        if (!session()->has('resetting_password_user_id')) {
            return redirect()->route('password.forgot')->with('error', 'Please request a password reset first.');
        }

        $user = User::find(session('resetting_password_user_id'));
        
        if (!$user) {
            session()->forget('resetting_password_user_id');
            return redirect()->route('password.forgot')->with('error', 'User not found. Please request a new password reset.');
        }

        // Generate secure 6-digit OTP (avoid sequential and repeated patterns)
        do {
            $otp = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (
            // Avoid sequential patterns (123456, 234567, etc.)
            preg_match('/(012345|123456|234567|345678|456789|567890|098765|987654|876543|765432|654321|543210)/', $otp) ||
            // Avoid repeated digits (000000, 111111, etc.)
            preg_match('/(\d)\1{5}/', $otp) ||
            // Avoid simple patterns (112233, 121212, etc.)
            preg_match('/(\d)\1{2}(\d)\2{2}/', $otp) ||
            preg_match('/(\d)(\d)\1\2\1\2/', $otp)
        );
        $otpExpiresAt = Carbon::now()->addMinutes(15);

        $user->update([
            'password_reset_otp' => $otp,
            'password_reset_otp_expires_at' => $otpExpiresAt,
        ]);

        // Send password reset OTP email
        try {
            Mail::to($user->email)->send(new PasswordResetEmail($user, $otp));
            \Log::info('Password reset OTP resent successfully to: ' . $user->email);
        } catch (\Exception $e) {
            \Log::error('Failed to resend password reset OTP email to ' . $user->email . ': ' . $e->getMessage());
            \Log::error('Email error trace: ' . $e->getTraceAsString());
            return back()->with('error', 'Failed to resend password reset code. Please try again later.');
        }

        // Update session
        $request->session()->put('last_password_reset_otp', $otp);
        $request->session()->put('password_reset_otp_expires_at', $otpExpiresAt->toDateTimeString());

        return back()->with('success', 'Password reset code has been resent to your email address.');
    }
}
