<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Mail\MarketingEmail;
use App\Models\NewsletterSubscriber;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class EmailMarketingController extends Controller
{
    public function index()
    {
        $subscribers = NewsletterSubscriber::where('is_active', true)
            ->orderBy('subscribed_at', 'desc')
            ->paginate(20);

        $totalSubscribers = NewsletterSubscriber::where('is_active', true)->count();
        $totalInactive = NewsletterSubscriber::where('is_active', false)->count();

        return view('admin.email-marketing.index', compact('subscribers', 'totalSubscribers', 'totalInactive'));
    }

    public function sendMarketingEmail(Request $request)
    {
        $validated = $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:10000',
        ]);

        $subscribers = NewsletterSubscriber::where('is_active', true)->get();
        $siteTitle = Setting::get('site_title', 'Vibrant Vogue');

        if ($subscribers->isEmpty()) {
            return back()->with('error', 'No active subscribers found.');
        }

        $sentCount = 0;
        $failedCount = 0;
        $errorMessages = [];

        // Send emails to all subscribers with proper headers
        foreach ($subscribers as $subscriber) {
            try {
                Mail::to($subscriber->email)->send(
                    new MarketingEmail($validated['subject'], $validated['message'], $siteTitle, $subscriber->email)
                );
                $sentCount++;
                
                // Add small delay to avoid rate limiting
                if ($sentCount % 10 == 0) {
                    usleep(500000); // 0.5 second delay every 10 emails
                }
            } catch (\Exception $e) {
                Log::error('Failed to send marketing email to ' . $subscriber->email . ': ' . $e->getMessage());
                Log::error('Email error trace: ' . $e->getTraceAsString());
                $errorMessages[] = 'Failed to send to ' . $subscriber->email . ': ' . $e->getMessage();
                $failedCount++;
            }
        }

        if ($sentCount > 0) {
            $message = "Marketing email sent successfully to {$sentCount} subscriber(s).";
            if ($failedCount > 0) {
                $message .= " {$failedCount} failed.";
            }
            return back()->with('success', $message);
        } else {
            return back()->with('error', 'Failed to send marketing emails. Please check your email configuration.')->withInput();
        }
    }

    public function toggleStatus($id)
    {
        $subscriber = NewsletterSubscriber::findOrFail($id);
        $subscriber->is_active = !$subscriber->is_active;
        $subscriber->save();

        return back()->with('success', 'Subscriber status updated successfully.');
    }

    public function destroy($id)
    {
        $subscriber = NewsletterSubscriber::findOrFail($id);
        $subscriber->delete();

        return back()->with('success', 'Subscriber deleted successfully.');
    }
}

