<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use App\Models\Product;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;

class DashboardController extends Controller
{
    public function index()
    {
        // Basic stats
        $stats = [
            'sliders' => Slider::count(),
            'active_sliders' => Slider::where('is_active', true)->count(),
            'products' => Product::count(),
            'active_products' => Product::where('is_active', true)->count(),
            'orders' => Order::count(),
            'users' => User::count(),
        ];

        // Revenue statistics
        // Total Revenue: All orders ever (paid + pending + refunded)
        $totalRevenue = Order::sum('total');
        
        // Paid Revenue: Orders with payment_status = 'paid' OR status = 'completed'
        // (Completed orders are considered paid even if payment_status is pending)
        $paidRevenue = Order::where(function($query) {
            $query->where('payment_status', 'paid')
                  ->orWhere('status', 'completed');
        })->sum('total');
        
        // Unpaid Revenue: Orders with payment_status = 'pending' AND status != 'completed'
        // (Completed orders are considered paid even if payment_status is pending)
        $pendingRevenue = Order::where('payment_status', 'pending')
            ->where('status', '!=', 'completed')
            ->sum('total');
        
        // Debug: Let's also check what payment statuses exist
        // $paymentStatuses = Order::selectRaw('payment_status, COUNT(*) as count, SUM(total) as total')->groupBy('payment_status')->get();
        
        // Today's Revenue: All orders created today (regardless of payment status)
        $todayStart = now()->startOfDay();
        $todayEnd = now()->endOfDay();
        $todayRevenue = Order::whereBetween('created_at', [$todayStart, $todayEnd])->sum('total');
        
        // Today's Paid Revenue: Paid orders OR completed orders created today
        $todayPaidRevenue = Order::whereBetween('created_at', [$todayStart, $todayEnd])
            ->where(function($query) {
                $query->where('payment_status', 'paid')
                      ->orWhere('status', 'completed');
            })
            ->sum('total');
        
        // This Month Revenue: All orders this month (regardless of payment status)
        $monthStart = now()->startOfMonth();
        $monthEnd = now()->endOfMonth();
        $thisMonthRevenue = Order::whereBetween('created_at', [$monthStart, $monthEnd])->sum('total');
        
        // This Month Paid Revenue: Paid orders OR completed orders this month
        $thisMonthPaidRevenue = Order::whereBetween('created_at', [$monthStart, $monthEnd])
            ->where(function($query) {
                $query->where('payment_status', 'paid')
                      ->orWhere('status', 'completed');
            })
            ->sum('total');

        // Order statistics
        $pendingOrders = Order::where('status', 'pending')->count();
        $processingOrders = Order::where('status', 'processing')->count();
        $completedOrders = Order::where('status', 'completed')->count();
        $cancelledOrders = Order::where('status', 'cancelled')->count();

        // Recent orders
        $recentOrders = Order::with('user')
            ->latest()
            ->limit(10)
            ->get();

        // Top selling products (from order items)
        $topProducts = \App\Models\OrderItem::selectRaw('product_id, product_name, SUM(quantity) as total_sold, SUM(subtotal) as total_revenue')
            ->groupBy('product_id', 'product_name')
            ->orderBy('total_sold', 'desc')
            ->limit(5)
            ->get();

        // Sales chart data (last 7 days)
        $salesData = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $salesData[] = [
                'date' => $date->format('M d'),
                'revenue' => Order::where('payment_status', 'paid')
                    ->whereDate('created_at', $date)
                    ->sum('total'),
                'orders' => Order::whereDate('created_at', $date)->count(),
            ];
        }

        return view('admin.dashboard', compact(
            'stats',
            'totalRevenue',
            'paidRevenue',
            'pendingRevenue',
            'todayRevenue',
            'todayPaidRevenue',
            'thisMonthRevenue',
            'thisMonthPaidRevenue',
            'pendingOrders',
            'processingOrders',
            'completedOrders',
            'cancelledOrders',
            'recentOrders',
            'topProducts',
            'salesData'
        ));
    }

    /**
     * Download QR code for the website
     */
    public function downloadQrCode()
    {
        // Get the website URL
        $websiteUrl = config('app.url');
        
        // Try to use SimpleSoftwareIO QR Code package if available
        if (class_exists('SimpleSoftwareIO\QrCode\Facades\QrCode')) {
            $qrCode = \SimpleSoftwareIO\QrCode\Facades\QrCode::format('png')
                ->size(500)
                ->generate($websiteUrl);
            
            return Response::make($qrCode, 200, [
                'Content-Type' => 'image/png',
                'Content-Disposition' => 'attachment; filename="website-qrcode.png"',
            ]);
        }
        
        // Fallback: Use QR Server API
        $size = 500;
        $qrCodeUrl = "https://api.qrserver.com/v1/create-qr-code/?size={$size}x{$size}&data=" . urlencode($websiteUrl);
        
        // Download the QR code from the API
        $qrCodeImage = file_get_contents($qrCodeUrl);
        
        if ($qrCodeImage === false) {
            return redirect()->route('admin.dashboard')
                ->with('error', 'Failed to generate QR code. Please try again later.');
        }
        
        return Response::make($qrCodeImage, 200, [
            'Content-Type' => 'image/png',
            'Content-Disposition' => 'attachment; filename="website-qrcode.png"',
        ]);
    }

    public function stores(Request $request)
    {
        $stores = Store::all();
        return view('admin.stores', compact('stores'));
    }
}
