<?php
/**
 * Fix .htaccess File
 * Upload to server root and visit: https://vibrantvogue.shop/FIX_HTACCESS.php
 * This will automatically fix your .htaccess file
 */

$htaccessPath = __DIR__ . '/public/.htaccess';

echo "<h1>Fix .htaccess File</h1>";
echo "<style>body{font-family:monospace;padding:20px;} .ok{color:green;} .error{color:red;} .info{color:blue;} pre{background:#f5f5f5;padding:10px;border:1px solid #ddd;}</style>";

// The correct .htaccess content
$correctContent = '<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options +FollowSymLinks -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]

    # Don\'t rewrite storage requests - let Apache serve files directly through symlink
    # This MUST be before the front controller rule
    RewriteCond %{REQUEST_URI} ^/storage/
    RewriteRule ^ - [L]
    
    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>';

// Check if file exists
if (!file_exists($htaccessPath)) {
    echo "<p class='error'>✗ .htaccess file NOT FOUND at: {$htaccessPath}</p>";
    echo "<p class='info'>Creating new file...</p>";
    
    // Create directory if it doesn't exist
    $dir = dirname($htaccessPath);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
} else {
    echo "<p class='ok'>✓ .htaccess file exists</p>";
    
    // Show current content
    $currentContent = file_get_contents($htaccessPath);
    echo "<h2>Current Content:</h2>";
    echo "<pre>" . htmlspecialchars($currentContent) . "</pre>";
}

// Check if we should fix it
$shouldFix = true;
if (file_exists($htaccessPath)) {
    $currentContent = file_get_contents($htaccessPath);
    if (strpos($currentContent, 'FollowSymLinks') !== false && 
        strpos($currentContent, 'RewriteCond %{REQUEST_URI} ^/storage/') !== false) {
        echo "<p class='ok'>✓ .htaccess already has correct content!</p>";
        $shouldFix = false;
    }
}

if ($shouldFix) {
    // Backup old file
    if (file_exists($htaccessPath)) {
        $backupPath = $htaccessPath . '.backup.' . date('Y-m-d_H-i-s');
        copy($htaccessPath, $backupPath);
        echo "<p class='info'>Backed up old file to: " . basename($backupPath) . "</p>";
    }
    
    // Write correct content
    if (file_put_contents($htaccessPath, $correctContent)) {
        echo "<p class='ok'>✓ .htaccess file fixed successfully!</p>";
        
        // Verify
        $newContent = file_get_contents($htaccessPath);
        echo "<h2>New Content:</h2>";
        echo "<pre>" . htmlspecialchars($newContent) . "</pre>";
        
        echo "<h2>Verification:</h2>";
        if (strpos($newContent, 'FollowSymLinks') !== false) {
            echo "<p class='ok'>✓ FollowSymLinks found</p>";
        } else {
            echo "<p class='error'>✗ FollowSymLinks NOT found</p>";
        }
        
        if (strpos($newContent, 'RewriteCond %{REQUEST_URI} ^/storage/') !== false) {
            echo "<p class='ok'>✓ Storage exclusion rule found</p>";
        } else {
            echo "<p class='error'>✗ Storage exclusion rule NOT found</p>";
        }
        
        echo "<hr>";
        echo "<p class='ok'><strong>✓ .htaccess file has been fixed!</strong></p>";
        echo "<p>Now test your images: <a href='/storage/products/'>Test Storage</a></p>";
    } else {
        echo "<p class='error'>✗ Failed to write .htaccess file!</p>";
        echo "<p>Check file permissions. The file should be writable.</p>";
        echo "<p>Run: <code>chmod 644 public/.htaccess</code></p>";
    }
} else {
    echo "<p class='ok'>No fix needed - file is already correct!</p>";
}

echo "<hr>";
echo "<h2>Manual Fix (if automatic fix failed):</h2>";
echo "<p>1. SSH into your server</p>";
echo "<p>2. Edit the file: <code>nano public/.htaccess</code></p>";
echo "<p>3. Replace the entire content with:</p>";
echo "<pre>" . htmlspecialchars($correctContent) . "</pre>";
echo "<p>4. Save: Ctrl+X, then Y, then Enter</p>";

