<?php
/**
 * Production Image Diagnostic Script
 * Upload this to your server root and access via browser:
 * https://vibrantvogue.shop/DIAGNOSE_PRODUCTION.php
 */

// Load Laravel
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle($request = Illuminate\Http\Request::capture());

echo "<h1>Production Image Diagnostic</h1>";
echo "<style>body{font-family:monospace;padding:20px;} .ok{color:green;} .error{color:red;} .info{color:blue;} pre{background:#f5f5f5;padding:10px;border:1px solid #ddd;}</style>";

// 1. Check APP_ENV
echo "<h2>1. Environment</h2>";
$appEnv = config('app.env');
$appUrl = config('app.url');
echo "<p>APP_ENV: <strong>{$appEnv}</strong></p>";
echo "<p>APP_URL: <strong>{$appUrl}</strong></p>";

// 2. Check storage symlink
echo "<h2>2. Storage Symlink</h2>";
$symlinkPath = public_path('storage');
$targetPath = storage_path('app/public');
if (is_link($symlinkPath)) {
    $linkTarget = readlink($symlinkPath);
    echo "<p class='ok'>✓ Symlink exists: <code>{$symlinkPath}</code></p>";
    echo "<p>Points to: <code>{$linkTarget}</code></p>";
    if (file_exists($linkTarget)) {
        echo "<p class='ok'>✓ Target exists</p>";
    } else {
        echo "<p class='error'>✗ Target does NOT exist!</p>";
    }
} else {
    echo "<p class='error'>✗ Symlink does NOT exist at: <code>{$symlinkPath}</code></p>";
    if (is_dir($symlinkPath)) {
        echo "<p class='error'>✗ It's a directory, not a symlink!</p>";
    }
}

// 3. Check if files exist
echo "<h2>3. Image Files</h2>";
$productsDir = storage_path('app/public/products');
if (is_dir($productsDir)) {
    $files = glob($productsDir . '/*.{jpg,jpeg,png,gif}', GLOB_BRACE);
    echo "<p class='ok'>✓ Products directory exists</p>";
    echo "<p>Found " . count($files) . " image files</p>";
    if (count($files) > 0) {
        $sampleFile = basename($files[0]);
        echo "<p>Sample file: <code>{$sampleFile}</code></p>";
        
        // Check if accessible via symlink
        $symlinkFile = public_path('storage/products/' . $sampleFile);
        if (file_exists($symlinkFile)) {
            echo "<p class='ok'>✓ File accessible via symlink</p>";
        } else {
            echo "<p class='error'>✗ File NOT accessible via symlink at: <code>{$symlinkFile}</code></p>";
        }
    }
} else {
    echo "<p class='error'>✗ Products directory does NOT exist: <code>{$productsDir}</code></p>";
}

// 4. Test helper function
echo "<h2>4. Helper Function Test</h2>";
if (function_exists('productImageUrl')) {
    echo "<p class='ok'>✓ Helper function exists</p>";
    
    // Get a product with image
    try {
        $product = \App\Models\Product::whereNotNull('image')->first();
        if ($product) {
            echo "<p>Product: <strong>{$product->name}</strong></p>";
            echo "<p>DB Image Path: <code>{$product->image}</code></p>";
            
            $generatedUrl = productImageUrl($product->image);
            echo "<p>Generated URL: <code>{$generatedUrl}</code></p>";
            
            // Check if URL is absolute
            if (strpos($generatedUrl, 'http') === 0) {
                echo "<p class='ok'>✓ URL is absolute</p>";
            } else {
                echo "<p class='error'>✗ URL is relative!</p>";
            }
            
            // Check if it uses /storage/ or /public/storage/
            if (strpos($generatedUrl, '/storage/') !== false && strpos($generatedUrl, '/public/storage/') === false) {
                echo "<p class='ok'>✓ Uses /storage/ path (correct for production)</p>";
            } elseif (strpos($generatedUrl, '/public/storage/') !== false) {
                echo "<p class='error'>✗ Uses /public/storage/ (wrong for production!)</p>";
            }
        } else {
            echo "<p class='error'>✗ No products with images found</p>";
        }
    } catch (\Exception $e) {
        echo "<p class='error'>✗ Error: {$e->getMessage()}</p>";
    }
} else {
    echo "<p class='error'>✗ Helper function does NOT exist!</p>";
    echo "<p>Run: <code>composer dump-autoload</code></p>";
}

// 5. Test direct file access
echo "<h2>5. Direct File Access Test</h2>";
if (isset($sampleFile)) {
    $testUrl = rtrim($appUrl, '/') . '/storage/products/' . $sampleFile;
    echo "<p>Test URL: <a href='{$testUrl}' target='_blank'>{$testUrl}</a></p>";
    echo "<p>Click the link above - if it shows 404, the .htaccess rule isn't working</p>";
}

// 6. Check .htaccess
echo "<h2>6. .htaccess Check</h2>";
$htaccessPath = public_path('.htaccess');
if (file_exists($htaccessPath)) {
    $htaccessContent = file_get_contents($htaccessPath);
    if (strpos($htaccessContent, 'RewriteCond %{REQUEST_URI} ^/storage/') !== false) {
        echo "<p class='ok'>✓ .htaccess has storage exclusion rule</p>";
    } else {
        echo "<p class='error'>✗ .htaccess missing storage exclusion rule!</p>";
    }
    if (strpos($htaccessContent, 'FollowSymLinks') !== false) {
        echo "<p class='ok'>✓ .htaccess has FollowSymLinks</p>";
    } else {
        echo "<p class='error'>✗ .htaccess missing FollowSymLinks!</p>";
    }
} else {
    echo "<p class='error'>✗ .htaccess does NOT exist!</p>";
}

// 7. Check permissions
echo "<h2>7. Permissions</h2>";
if (is_dir($productsDir)) {
    $perms = substr(sprintf('%o', fileperms($productsDir)), -4);
    echo "<p>Products directory permissions: <code>{$perms}</code></p>";
    if ($perms >= '0755') {
        echo "<p class='ok'>✓ Permissions OK</p>";
    } else {
        echo "<p class='error'>✗ Permissions too restrictive!</p>";
    }
}

echo "<hr>";
echo "<h2>Summary</h2>";
echo "<p>If images still don't work, check:</p>";
echo "<ol>";
echo "<li>Symlink exists and points to correct location</li>";
echo "<li>Files exist in storage/app/public/products/</li>";
echo "<li>.htaccess has storage exclusion rule</li>";
echo "<li>Helper function generates correct URLs (using /storage/, not /public/storage/)</li>";
echo "<li>Direct file access works (click test URL above)</li>";
echo "</ol>";

