<?php
/**
 * Diagnostic script to check image URL generation
 * Run this via: php DIAGNOSE_IMAGE_URLS.php
 */

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

echo "=== IMAGE URL DIAGNOSTIC ===\n\n";

// 1. Check APP_URL
echo "1. APP_URL Configuration:\n";
echo "   APP_URL from .env: " . env('APP_URL', 'NOT SET') . "\n";
echo "   APP_URL from config: " . config('app.url', 'NOT SET') . "\n";
echo "   Current request URL: " . (isset($_SERVER['HTTP_HOST']) ? 'http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] : 'N/A') . "\n\n";

// 2. Check filesystems config
echo "2. Filesystem Configuration:\n";
$publicDisk = config('filesystems.disks.public');
echo "   Public disk URL: " . ($publicDisk['url'] ?? 'NOT SET') . "\n";
echo "   Public disk root: " . ($publicDisk['root'] ?? 'NOT SET') . "\n\n";

// 3. Test Storage::url()
echo "3. Storage::url() Test:\n";
try {
    $testUrl = Storage::disk('public')->url('products/test.jpg');
    echo "   Storage::url('products/test.jpg'): " . $testUrl . "\n";
} catch (\Exception $e) {
    echo "   ERROR: " . $e->getMessage() . "\n";
}
echo "\n";

// 4. Test productImageUrl helper
echo "4. productImageUrl() Helper Test:\n";
if (function_exists('productImageUrl')) {
    echo "   ✅ Helper function exists\n";
    try {
        $testUrl = productImageUrl('products/test.jpg');
        echo "   productImageUrl('products/test.jpg'): " . $testUrl . "\n";
        
        $testUrl2 = productImageUrl('/storage/products/test.jpg');
        echo "   productImageUrl('/storage/products/test.jpg'): " . $testUrl2 . "\n";
        
        $testUrl3 = productImageUrl('/public/storage/products/test.jpg');
        echo "   productImageUrl('/public/storage/products/test.jpg'): " . $testUrl3 . "\n";
    } catch (\Exception $e) {
        echo "   ERROR: " . $e->getMessage() . "\n";
    }
} else {
    echo "   ❌ Helper function does NOT exist\n";
}
echo "\n";

// 5. Check actual product from database
echo "5. Real Product Image Test:\n";
try {
    $product = \App\Models\Product::whereNotNull('image')->first();
    if ($product) {
        echo "   Product: " . $product->name . "\n";
        echo "   DB Image Path: " . $product->image . "\n";
        echo "   Generated URL: " . productImageUrl($product->image) . "\n";
        
        // Check if file actually exists
        $filePath = storage_path('app/public/' . ltrim(str_replace(['/storage/', '/public/storage/', 'storage/', 'public/storage/'], '', $product->image), '/'));
        echo "   File exists check: " . (file_exists($filePath) ? "✅ YES" : "❌ NO") . " at: " . $filePath . "\n";
    } else {
        echo "   No products with images found in database\n";
    }
} catch (\Exception $e) {
    echo "   ERROR: " . $e->getMessage() . "\n";
}
echo "\n";

// 6. Check storage symlink
echo "6. Storage Symlink Check:\n";
$symlinkPath = public_path('storage');
if (is_link($symlinkPath)) {
    echo "   ✅ Symlink exists: " . $symlinkPath . "\n";
    echo "   Points to: " . readlink($symlinkPath) . "\n";
    echo "   Target exists: " . (file_exists(readlink($symlinkPath)) ? "✅ YES" : "❌ NO") . "\n";
} elseif (is_dir($symlinkPath)) {
    echo "   ⚠️  public/storage is a directory, not a symlink\n";
} else {
    echo "   ❌ public/storage does not exist\n";
}
echo "\n";

echo "=== END DIAGNOSTIC ===\n";

