<?php
/**
 * Image System Diagnostic Script
 * Run this on your server to diagnose image issues
 * 
 * Usage: php DIAGNOSE_IMAGES.php
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "=== IMAGE SYSTEM DIAGNOSTIC ===\n\n";

// 1. Check if helper is loaded
echo "1. Checking if productImageUrl helper is loaded...\n";
if (function_exists('productImageUrl')) {
    echo "   ✅ Helper function exists\n";
} else {
    echo "   ❌ Helper function NOT FOUND - Run: composer dump-autoload\n";
    exit(1);
}

// 2. Test helper with different path formats
echo "\n2. Testing helper with different path formats...\n";
$testPaths = [
    'products/test.jpg',
    'storage/products/test.jpg',
    '/storage/products/test.jpg',
    '/public/storage/products/test.jpg',
    'public/storage/products/test.jpg',
    'https://example.com/image.jpg',
    ''
];

foreach ($testPaths as $path) {
    $result = productImageUrl($path);
    echo "   Input: '{$path}' → Output: '{$result}'\n";
}

// 3. Check storage link
echo "\n3. Checking storage symlink...\n";
$storageLink = public_path('storage');
if (is_link($storageLink)) {
    $target = readlink($storageLink);
    echo "   ✅ Symlink exists: {$storageLink} → {$target}\n";
    if (file_exists($storageLink)) {
        echo "   ✅ Symlink target exists\n";
    } else {
        echo "   ❌ Symlink target does NOT exist: {$target}\n";
    }
} elseif (is_dir($storageLink)) {
    echo "   ⚠️  {$storageLink} is a directory, not a symlink\n";
    echo "   Run: rm -rf public/storage && php artisan storage:link\n";
} else {
    echo "   ❌ Symlink does NOT exist: {$storageLink}\n";
    echo "   Run: php artisan storage:link\n";
}

// 4. Check storage directories
echo "\n4. Checking storage directories...\n";
$dirs = ['products', 'categories', 'sliders', 'banners', 'settings'];
foreach ($dirs as $dir) {
    $path = storage_path("app/public/{$dir}");
    if (is_dir($path)) {
        $count = count(glob($path . '/*'));
        echo "   ✅ {$dir}/ exists ({$count} files)\n";
    } else {
        echo "   ⚠️  {$dir}/ does NOT exist\n";
    }
}

// 5. Check APP_URL
echo "\n5. Checking APP_URL configuration...\n";
$appUrl = config('app.url');
echo "   APP_URL: {$appUrl}\n";
if (empty($appUrl) || $appUrl === 'http://localhost') {
    echo "   ⚠️  APP_URL not set correctly in .env\n";
    echo "   Set: APP_URL=https://vibrantvogue.shop\n";
} else {
    echo "   ✅ APP_URL is set\n";
}

// 6. Test Storage::url()
echo "\n6. Testing Storage::url()...\n";
try {
    $testUrl = Storage::disk('public')->url('products/test.jpg');
    echo "   Storage::url('products/test.jpg'): {$testUrl}\n";
    echo "   ✅ Storage::url() works\n";
} catch (\Exception $e) {
    echo "   ❌ Storage::url() failed: " . $e->getMessage() . "\n";
}

// 7. Check a real product image
echo "\n7. Checking real product images from database...\n";
try {
    $products = \App\Models\Product::whereNotNull('image')->take(3)->get(['id', 'name', 'image']);
    if ($products->count() > 0) {
        foreach ($products as $product) {
            $url = productImageUrl($product->image);
            echo "   Product: {$product->name}\n";
            echo "   DB Path: {$product->image}\n";
            echo "   Generated URL: {$url}\n";
            echo "\n";
        }
    } else {
        echo "   ⚠️  No products with images found\n";
    }
} catch (\Exception $e) {
    echo "   ❌ Error checking products: " . $e->getMessage() . "\n";
}

echo "\n=== DIAGNOSTIC COMPLETE ===\n";
echo "\nIf images still don't work:\n";
echo "1. Run: composer dump-autoload\n";
echo "2. Run: php artisan config:clear\n";
echo "3. Run: php artisan cache:clear\n";
echo "4. Run: php artisan storage:link\n";
echo "5. Check .env has: APP_URL=https://vibrantvogue.shop\n";

